import 'dart:convert';
import 'dart:io';
import 'package:open_file/open_file.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:viserpay_merchant/core/utils/url_container.dart';
import 'package:http/http.dart' as http;
import '../../../core/utils/my_color.dart';
import '../../../core/utils/my_strings.dart';
import '../../../view/components/snack_bar/show_custom_snackbar.dart';
import '../../model/global/response_model/response_model.dart';
import '../../model/invoice/invoice_history_response_model.dart';
import '../../repo/invoice/invoice_history_repo.dart';

class InvoiceHistoryController extends GetxController {
  InvoiceHistoryRepo invoiceHistoryRepo;
  InvoiceHistoryController({required this.invoiceHistoryRepo});

  bool isLoading = true;
  InvoiceHistoryResponseModel model = InvoiceHistoryResponseModel();

  int page = 0;
  String? nextPageUrl;

  List<InvoiceDataModel> invoiceList = [];

  Future<void> initialData() async {
    page = 0;
    invoiceList.clear();
    isLoading = true;
    update();

    await loadInvoiceData();
    isLoading = false;
    update();
  }

  Future<void> loadInvoiceData() async {
    page = page + 1;
    if (page == 1) {
      invoiceList.clear();
    }

    ResponseModel responseModel = await invoiceHistoryRepo.getAllInvoiceData(page);
    // if(  responseModel.statusCode == 200){
    if (200 == 200) {
      model = InvoiceHistoryResponseModel.fromJson(jsonDecode(responseModel.responseJson));

      nextPageUrl = model.data?.invoices?.nextPageUrl;
      if (model.status.toString().toLowerCase() == MyStrings.success.toLowerCase()) {
        List<InvoiceDataModel>? tempInvoiceList = model.data?.invoices?.data;
        if (tempInvoiceList != null && tempInvoiceList.isNotEmpty) {
          invoiceList.addAll(tempInvoiceList);
        }
      } else {
        CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.somethingWentWrong]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message]);
    }

    isLoading = false;
    update();
  }

  bool hasNext() {
    return nextPageUrl != null && nextPageUrl!.isNotEmpty && nextPageUrl != 'null' ? true : false;
  }

  dynamic getStatusOrColor(int index, {bool isStatus = true}) {
    String status = invoiceList[index].status ?? '';

    if (isStatus) {
      String text = status == "0"
          ? MyStrings.unpaid
          : status == "1"
              ? MyStrings.paid
              : status == "2"
                  ? MyStrings.rejected
                  : status == "3"
                      ? MyStrings.canceled
                      : "";
      return text;
    } else {
      Color color = status == "0"
          ? MyColor.highPriorityPurpleColor
          : status == "1"
              ? MyColor.colorGreen
              : status == "2"
                  ? MyColor.colorRed
                  : MyColor.colorOrange;
      return color;
    }
  }

  //DOWNLOAD

  String downloadUrl = "";
  String downloadFileName = "";
  bool downloadLoading = false;
  Future<void> downloadINVOICE(String invoiceID) async {
    downloadLoading = true;
    update();
    final headers = {
      'Authorization': "Bearer ${invoiceHistoryRepo.apiClient.token}",
      'content-type': 'application/pdf',
    };

    String url = "${UrlContainer.baseUrl}${UrlContainer.invoiceDownload}/$invoiceID";
    http.Response response = await http.get(Uri.parse(url), headers: headers);
    final bytesSS = response.body;
    final bytes = response.bodyBytes;
    print(bytesSS);
    print(bytes);
    String extension = "pdf";

    await saveAndOpenFile(bytes, '${MyStrings.appName}_${DateTime.now().millisecondsSinceEpoch}.$extension', extension);

    return;
  }

  Future<void> saveAndOpenFile(List<int> bytes, String fileName, String extension) async {
    Directory? downloadsDirectory;

    if (Platform.isAndroid) {
      await Permission.storage.request();
      downloadsDirectory = Directory('/storage/emulated/0/Download');
    } else if (Platform.isIOS) {
      downloadsDirectory = await getApplicationDocumentsDirectory();
    }

    if (downloadsDirectory != null) {
      final downloadPath = '${downloadsDirectory.path}/$fileName';
      final file = File(downloadPath);
      await file.writeAsBytes(bytes);
      CustomSnackBar.success(successList: ['File saved at: $downloadPath']);
      print('File saved at: $downloadPath');
      await openFile(downloadPath, extension);
    } else {
      CustomSnackBar.error(errorList: ["error"]);
    }
    downloadLoading = false;
    update();
  }

  Future<void> openFile(String path, String extension) async {
    final file = File(path);
    if (await file.exists()) {
      final result = await OpenFile.open(path);
      if (result.type != ResultType.done) {
        if (result.type == ResultType.noAppToOpen) {
          CustomSnackBar.error(errorList: ['File saved at: $path']);
          // CustomSnackBar.error(errorList: [MyStrings.noDocOpenerApp, 'File saved at: $path']);
        }
      }
    } else {
      CustomSnackBar.error(errorList: [MyStrings.fileNotFound]);
    }
  }
}
